<?php

namespace App\Http\Controllers\Timelog;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Box\Spout\Writer\Common\Creator\WriterEntityFactory;
use Box\Spout\Writer\Common\Creator\Style\StyleBuilder;
use Box\Spout\Common\Entity\Style\Color;

// use OpenSpout\Writer\Common\Creator\Style\StyleBuilder;
// use OpenSpout\Common\Entity\Style\Color;

use App\Models\User;
use App\Models\Timelog;
use App\Models\TimelogBreak;
use Illuminate\Support\Facades\Auth;

class TimelogController extends Controller
{

    // public function __construct()
    // {
    //     $this->middleware('auth');
    // }

    public static function apiRoutes()
    {
        Route::middleware('auth')->group(function () {
            // Route::get('timelog', [TimelogController::class, 'index']);

            Route::get('timelog', [TimelogController::class, 'index']);
            // Timelog Process
            Route::get('/timelog/today/list', [TimelogController::class, 'today_timelog']);
            Route::post('/timelog/time-in/store', [TimelogController::class, 'time_in']);
            Route::put('/timelog/time-out/update/{id}', [TimelogController::class, 'time_out']);
            // Break Process
            Route::post('/timelog/start-break/store', [TimelogController::class, 'start_break']);
            Route::put('/timelog/end-break/update/{id}', [TimelogController::class, 'end_break']);
            // Timelog Summary
            Route::get('/timelog/user/summary', [TimelogController::class, 'summary']);
            Route::get('/timelog/summary/list', [TimelogController::class, 'summary_list']);
            Route::get('/timelog/summary/list/{from}/{to}', [TimelogController::class, 'summary_list']);
            // Timelog Statistics
            Route::get('/timelog/statistics/list', [TimelogController::class, 'statistics']);
            // Timelog admin summary report
            Route::post('/timelog/export/report', [TimelogController::class, 'export_summary_list']);
            Route::get('/reports/admin/timelog-summary', [TimelogController::class, 'admin_summary']);
            Route::get('/timelog/admin/summary/list/{userid}', [TimelogController::class, 'admin_summary_list']);
            Route::get('/timelog/admin/summary/list/{userid}/{admin_from}/{admin_to}', [TimelogController::class, 'admin_summary_list']);
            // Download Excel File
            Route::get('/reports/download/summary', [TimelogController::class, 'download_reports']);

        });
    }

    private function _hts($time)
    {
        list($h, $m, $s) = explode(':', $time);
        return ($h * 3600) + ($m * 60) + $s;
    }

    private function time_from_seconds($seconds) { 
        $h = floor($seconds / 3600);
        $m = floor(($seconds % 3600) / 60);
        $s = $seconds - ($h * 3600) - ($m * 60);
        return sprintf('%2d hr %2d mins', $h, $m, $s);
        // return sprintf('%02d:%02d:%02', $h, $m, $s);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if(Auth::user()->dtr_id){
            $dtr = DtrTimelogController::getCollection(Auth::user()->dtr_id);
            $timelogChecker = $dtr['timelogChecker'];
            $breaksChecker = $dtr['breaksChecker'];
        }else {
            // Timelog Checker if timelog exist
            $timelogChecker = Timelog::select(['id','user_id', 'time_in', 'time_out', 'total_mins_break'])->where([
                ['user_id', Auth::user()->id],
            ])->whereDate('time_in', Carbon::now()->format('Y-m-d'))->first();
    
    
            // Breaks Checker if timelog exist
            if ($timelogChecker != null) {
                # code...
                $breaksChecker = TimelogBreak::select(['timelog_id', 'break_start', 'break_end'])->where([
                    ['timelog_id', $timelogChecker->id],
                ])->whereDate('break_start', Carbon::now()->format('Y-m-d'))->orderBy('break_start', 'DESC')->first();
                
            }
        }
        

        \JavaScript::put([
            'timelogDetails' => $timelogChecker,
            'breakDetails' => $timelogChecker != null ? $breaksChecker : '',
        ]);

        return view('timelog.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    // Timelog Statistics
    public function statistics() {

        if (Auth::user()->dtr_id) {
            $displayStats = DtrTimelogController::statistics(Auth::user()->dtr_id);
        } else {
            Carbon::setWeekStartsAt(Carbon::MONDAY);
            Carbon::setWeekEndsAt(Carbon::FRIDAY);
            $from_date = Carbon::now()->startOfMonth()->shiftTimezone(Auth::user()->timezone);
            $to_date = Carbon::now()->setTimezone(Auth::user()->timezone);
            $tl = [];
            $abs_ctr = 0;;
            $timelogStats = Timelog::with('breaks')->where('user_id', Auth::user()->id)->whereMonth('time_in', Carbon::now()->month)->orderBy('created_at', 'DESC')->get();
            $user_timelog_week = Timelog::where('user_id', Auth::user()->id)->whereBetween('time_in', [Carbon::now()->startOfWeek()->shiftTimezone(Auth::user()->timezone), Carbon::now()->endOfWeek()->shiftTimezone(Auth::user()->timezone)])->get();
            $todayLogin = Timelog::where('user_id', Auth::user()->id)->orderBy('time_in', 'DESC')->first();
            $tardyCountWk = $tardyCountMos = $weeklyWork = 0;

            // Absent Checker
            while (strtotime($from_date) <= strtotime($to_date)) {
                $date = $from_date->format('Y-m-d');
                $day = $from_date->format('D');
                if (!in_array($day, ['Sat', 'Sun'])) {
                    $isPresent = 0;
                    foreach ($timelogStats as $key => $value) {
                        if ($date == Carbon::createFromFormat('Y-m-d H:i:s', $value->time_in)->shiftTimezone(Auth::user()->timezone)->format('Y-m-d')) {
                            $isPresent = 1;
                        }
                    }
                    if ($isPresent != 1) {
                        $tl[$date] = 'ABSENT';
                        if ($tl[$date] == 'ABSENT') {
                            $abs_ctr = $abs_ctr + 1;
                        }
                    }
                }
                $from_date = $from_date->addDay();
            }

            // Total Weekly Work Hour
            foreach ($user_timelog_week as $weekly_total) {
                if ($weekly_total->time_out != NULL) {
                    $total_diff_sec = $weekly_total->time_in->shiftTimezone(Auth::user()->timezone)->diffInSeconds($weekly_total->time_out->shiftTimezone(Auth::user()->timezone));
                } else {
                    $total_diff_sec = $weekly_total->time_in->shiftTimezone(Auth::user()->timezone)->diffInSeconds($to_date->format('H:i:s'));
                }
                $weeklyWork = ($total_diff_sec + $weeklyWork);
            }

            // Total Tardy Hours
            foreach ($timelogStats as $month_total) {
                $tardyCountMos = $month_total->total_mins_tardy + $tardyCountMos;
            }

            $displayStats = [
                'weekly_work_hours' => $this->time_from_seconds($weeklyWork),
                'lates' => $tardyCountMos,
                'absents' => $abs_ctr,
            ];

            // Today hours worked
            if (Carbon::parse($todayLogin->time_in)->format('Y-m-d') == $to_date->format('Y-m-d')) {
                $displayStats['today_work_hours'] = $this->time_from_seconds($todayLogin->time_in->shiftTimezone(Auth::user()->timezone)->diffInSeconds($to_date));
            }
        }
        
        return $displayStats;
    }

    // Timelog and Breaks
    public function today_timelog() {
        $timelogList = [];
        
        if(Auth::user()->dtr_id) {
            $todayTimelog = DtrTimelogController::todayTimelog(Auth::user()->dtr_id);
        }else {
            $todayTimelog = Timelog::with('breaks')->where('user_id', Auth::user()->id)->whereDate('created_at', Carbon::now()->format('Y-m-d'))->orderBy('created_at', 'DESC')->get(); 
        }
        foreach ($todayTimelog as $key => $value) {
            
            if($value->time_in) {
                $timelogList['id'] = $value->id;
                $timelogList['user_id'] = $value->user_id;
                $timelogList['time_in'] = isset($value->time_in) ? Carbon::parse($value->time_in)->format('Y-m-d H:i:s') : null;
                $timelogList['time_out'] = isset($value->time_out) ? Carbon::parse($value->time_out)->format('Y-m-d H:i:s') : null;
                $timelogList['total_mins_break'] = $value->total_mins_break;
                $timelogList['total_mins_tardy'] = $value->total_mins_tardy;
                foreach ($value->breaks as $k => $v) {
                    $timelogList['breaks'][] = [
                        'id' => $v->id,
                        'timelog_id' => $v->timelog_id,
                        'break_start' => isset($v->break_start) ? $v->break_start->format('Y-m-d H:i:s') : null,
                        'break_end' => isset($v->break_end) ? $v->break_end->format('Y-m-d H:i:s') : null,
                        'total_breaks' => $v->total_breaks,
                        'notes' => $v->notes,
                    ];
                }
            }
        }
        return $timelogList;
    }

    // Timelog
    public function time_in(Request $request){
        // dd(Carbon::now());
        if (Auth::user()->dtr_id) {
            $timelogProcess = DtrTimelogController::timeIn(Auth::user()->dtr_id);
        } else {
            $timelogProcess = new Timelog();
            $timein = Carbon::now()->setTimezone(Auth::user()->timezone);
            $logintime = Carbon::parse(Auth::user()->logintime)->shiftTimezone(Auth::user()->timezone);
            if ($timein > $logintime) {
                $total_mins_tardy = Carbon::parse($timein)->diffInMinutes($logintime);
            } else {
                $total_mins_tardy = 0;
            }

            $data = [
                'user_id' => Auth::user()->id,
                'time_in' => $timein,
                'total_mins_tardy' => $total_mins_tardy,
            ];

            $timelogProcess->fill($data);
            $timelogProcess->save();

            return $timelogProcess;
        }
    }

    public function time_out(Request $request, $id){
        if(Auth::user()->dtr_id) {
            $updateTimelog = DtrTimelogController::timeOut($id);
        }else {
            $updateTimelog = Timelog::where('id', $id)->whereDate('time_in', Carbon::now()->format('Y-m-d'))->first();
            $updateTimelog->time_out = Carbon::now()->setTimezone(Auth::user()->timezone)->format('Y-m-d H:i:s');
            $updateTimelog->update();
        }
        return $updateTimelog;
    }

    // Break
    public function start_break(Request $request) {
        $data = [
            'timelog_id' => $request->timelog_id,
            'break_start' => Carbon::now()->setTimezone(Auth::user()->timezone)->format('Y-m-d H:i:s'),
            'notes' => $request->notes,
        ];
        if(Auth::user()->dtr_id) {
            $breakProcess = DtrTimelogController::startBreak($data);
        }else {
            $breakProcess = new TimelogBreak();
            $breakProcess->fill($data);
            $breakProcess->save();
        }

        return $breakProcess;
    }
    
    public function end_break(Request $request, $id) {

        $break_end = Carbon::now()->setTimezone(Auth::user()->timezone)->format('Y-m-d H:i:s');

        if(Auth::user()->dtr_id) {
            $breaksChecker = DtrTimelogController::endBreak($id, $break_end);
        }else {
            $breaksChecker = TimelogBreak::with(['timelog'])->where([
                ['timelog_id', $id],
            ])->whereDate('break_start', Carbon::now()->format('Y-m-d'))->orderBy('break_start', 'DESC')->first();
    
            $totalPerBreaks = $breaksChecker->break_start->diffInMinutes($break_end);
    
            if ($breaksChecker->break_end === null) {
                $breaksChecker->break_end = $break_end;
                $breaksChecker->total_breaks = $totalPerBreaks;
                $breaksChecker->timelog['total_mins_break'] = $breaksChecker->timelog['total_mins_break'] + $totalPerBreaks;
                $breaksChecker->update();
                $breaksChecker->timelog->update();
            }
        }

        return $breaksChecker;
    }

    // Individual User Summary Report
    public function summary() {
        return view('timelog.summary');
    }

    public function summary_list($from=null, $to=null) {

        if (Auth::user()->dtr_id) {
            $findTimelog = DtrTimelogController::summary_list($from, $to, Auth::user()->dtr_id);
        } else {
            $findTimelog = Timelog::select(['id', 'user_id', 'time_in', 'time_out', 'total_mins_break', 'total_mins_tardy'])->where('user_id', Auth::user()->id);
            if ($from && $to) {
                $findTimelog = $findTimelog->whereBetween('time_in', [$from, Carbon::parse($to)->addDay(1)->format('Y-m-d')])->orderBy('time_in', 'DESC');
            } else {
                $findTimelog = $findTimelog->whereBetween('time_in', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->orderBy('time_in', 'DESC');
            }
        }

        return $findTimelog->get();

    }

    // Admin Summary Report
    public function admin_summary() {

        if (Auth::user()->dtr_id) {
            $slct_user = $this->getDtrUsers();
        } else {
            $slct_user = User::with('party')->where('is_active', 1)->whereNull('dtr_id')->orderBy('username', 'asc')->get();
        }


        if (Auth::user()->dtr_id) {
            $userArr = DtrTimelogController::admin_summary(Auth::user()->dtr_id);
        } else {
            $userArr = [];
            $allUser = User::with(['party'])->get();
            foreach ($allUser as $key_user => $value_user) {
                $userTimelog = Timelog::where('user_id', $value_user->id)->whereBetween('time_in', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->groupBy('user_id')->orderBy('time_in', 'DESC')->get();
                foreach ($userTimelog as $key_timelog => $value_timelog) {
                    $userArr[] = [
                        'user_id' => $value_user->id,
                        'user_name' => $value_user->party['name'],
                    ];
                }
            }
        }
        \JavaScript::put([
            'users' => $slct_user,
        ]);
        return view('reports.timelog-admin-summary-list');
    }

    public function admin_summary_list($userid, $admin_from=null, $admin_to=null) {

        $dtr_ids = [];

        if (Auth::user()->dtr_id) {
            $allUser = User::with('party')->get();
            foreach ($allUser as $k => $kuser) {
                if ($kuser->dtr_id) {
                    array_push($dtr_ids, $kuser->dtr_id);
                }
            }
            $reportTimelog = DtrTimelogController::admin_summary_list($userid, $admin_from, $admin_to, $dtr_ids);
        } else {
            
            $reportTimelog = [];

            if ($userid != 'all') {
                $allUser = User::where('id', $userid)->get();
            } else {
                $allUser = User::get();
            }
            foreach ($allUser as $key_user => $value_user) {
                $userTimelog = Timelog::where('user_id', $value_user->id);

                if ($admin_from && $admin_to) {
                    $userTimelog = $userTimelog->whereBetween('time_in', [$admin_from, Carbon::parse($admin_to)->addDay(1)->format('Y-m-d')])->orderBy('time_in', 'DESC')->get();
                } else {
                    $userTimelog = $userTimelog->whereBetween('time_in', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->orderBy('time_in', 'DESC')->get();
                }

                foreach ($userTimelog as $key_timelog => $value_timelog) {
                    $reportTimelog[] = [
                        'user_id' => $value_user->id,
                        'user_name' => $value_user->party['name'],
                        'date' => Carbon::parse($value_timelog->time_in)->format('M d, Y'),
                        'day' => Carbon::parse($value_timelog->time_in)->format('D'),
                        'in' => Carbon::parse($value_timelog->time_in)->shiftTimezone(Auth::user()->timezone)->format('H:i:s'),
                        'out' => $value_timelog->time_out ? Carbon::parse($value_timelog->time_out)->shiftTimezone(Auth::user()->timezone)->format('H:i:s') : null,
                        'time_in' => $value_timelog-> time_in->shiftTimezone(Auth::user()->timezone),
                        'time_out' => $value_timelog->time_out ? Carbon::parse($value_timelog->time_out)->shiftTimezone(Auth::user()->timezone) : null,
                        'hours_worked' => $value_timelog->time_out ? $value_timelog-> time_in->shiftTimezone(Auth::user()->timezone)->diffInSeconds($value_timelog->time_out->shiftTimezone(Auth::user()->timezone)) : null,
                        'total_mins_break' => $value_timelog->total_mins_break,
                        'total_mins_tardy' => $value_timelog->total_mins_tardy,
                    ];
                }
            }
            
        }
        return $reportTimelog;
    }

    public function export_summary_list(Request $request){
        $reportTimelog = [];
        $totals = [
            'total_worked_hours' => 0,
            'total_breaks' => 0,
            'total_tardy' => 0
        ];

        foreach ($request->data as $key_timelog => $value_timelog) {
            $reportTimelog[] = [
                $value_timelog['user_name'],
                $value_timelog['date'],
                $value_timelog['day'],
                $value_timelog['in'],
                $value_timelog['out'],
                self::time_from_seconds($value_timelog['hours_worked']),
                $value_timelog['total_mins_break'] . 'mins',
                $value_timelog['total_mins_tardy'] . 'mins',
            ];

            $totals['total_worked_hours'] += $value_timelog['hours_worked'];
            $totals['total_breaks'] += $value_timelog['total_mins_break'];
            $totals['total_tardy'] += $value_timelog['total_mins_tardy'];
        }

        $totals['total_worked_hours'] = $this->time_from_seconds($totals['total_worked_hours']);
        $totals['total_breaks'] = $totals['total_breaks'] . ' mins';
        $totals['total_tardy'] = $totals['total_tardy'] . ' mins';

        if (!File::isDirectory(storage_path('app/timelog-report'))) {
            File::makeDirectory(storage_path('app/timelog-report'), 0777, true, true);
        }

        /** Create a style with the StyleBuilder */
        $headerStyles = (new StyleBuilder())
            ->setFontBold()
            ->setFontSize(12)
            ->setFontColor(Color::WHITE)
            ->setShouldWrapText(false)
            ->setBackgroundColor(Color::rgb(240,90,41))
            ->build();

        $footerStyles = (new StyleBuilder())
            ->setFontBold()
            ->setFontSize(12)
            ->setFontColor(Color::BLACK)
            ->setShouldWrapText(false)
            ->setBackgroundColor(Color::YELLOW)
            ->build();

        $writer = WriterEntityFactory::createXLSXWriter();
        $writer->openToFile(storage_path('app/timelog-report') . '/reports.xlsx'); // write data to a file or to a PHP stream

        /* Header Row */
        $headerRow = WriterEntityFactory::createRowFromArray([
            'Staff Name', 
            'Date', 
            'Day', 
            'Time-in', 
            'Time-out', 
            'Hours Worked', 
            'Break', 
            'Lates'
        ], $headerStyles);

        /* Footer Row */
        $footerRow = WriterEntityFactory::createRowFromArray([
            null, 
            null, 
            null, 
            null, 
            'totals', 
            $totals['total_worked_hours'], 
            $totals['total_breaks'], 
            $totals['total_tardy']
        ], $footerStyles);

        /** add multiple rows at a time */
        // Data Row
        foreach ($reportTimelog as $key => $v) {
            $dataRow[] = WriterEntityFactory::createRowFromArray($v);
        }
        $writer->addRow($headerRow);
        $writer->addRows($dataRow);
        $writer->addRow($footerRow);
        $writer->close();
        exit();
    }

    public function getDtrUsers()
    {
        $dtr_ids = [];
        $allUser = User::with('party')->get();
        foreach ($allUser as $k => $kuser) {
            if ($kuser->dtr_id) {
                array_push($dtr_ids, $kuser->dtr_id);
            }
        }
        $slct_user = DtrTimelogController::getUsers($dtr_ids);
        return $slct_user;
    }

    public function download_reports()
    {
        $rename = 'generated-timelog-report-' . Carbon::now()->format('m-j-Y') . '.xlsx';
        $excelFile = storage_path('app/timelog-report') . '/reports.xlsx';
        $filetype = filetype($excelFile);
        header("Content-Type: " . $filetype);
        header("Content-Length: " . filesize($excelFile));
        header("Content-Disposition: attachment; filename=" . $rename);
        readfile($excelFile);
    }    
}
