<?php

namespace App\Http\Controllers\Timelog;

use App\Http\Controllers\Controller;
use App\Models\DTR\Timelog;
use App\Models\DTR\TimelogBreak;
use App\Models\DTR\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DtrTimelogController extends Controller
{

    private static function timezone($dtr_id)
    {
        $user = User::find($dtr_id);
        return $user->timezone;
    }

    private static function time_from_seconds($seconds)
    {
        $h = floor($seconds / 3600);
        $m = floor(($seconds % 3600) / 60);
        $s = $seconds - ($h * 3600) - ($m * 60);
        return sprintf('%2d hr %2d mins', $h, $m, $s);
        // return sprintf('%02d:%02d:%02', $h, $m, $s);
    }

    public static function getCollection($dtr_id)
    {
        // Timelog Checker if timelog exist
        $timelogChecker = Timelog::select([
            'id',
            'user_id',
            'time_in',
            'time_out',
            'total_mins_break'
        ])->where('user_id', $dtr_id)
        ->whereDate('time_in', Carbon::now()->format('Y-m-d'))
        ->first();


        // Breaks Checker if timelog exist
        if ($timelogChecker != null) {
            # code...
            $breaksChecker = TimelogBreak::select(['timelog_id', 'break_start', 'break_end'])->where([
                ['timelog_id', $timelogChecker->id],
            ])->whereDate('break_start', Carbon::now()->format('Y-m-d'))->orderBy('break_start', 'DESC')->first();
            
        }

        return [
            'timelogChecker' => $timelogChecker,
            'breaksChecker' => $breaksChecker ?? null
        ];
    }

    public static function todayTimelog($dtr_id)
    {
        $tl = Timelog::with('breaks')
        ->where('user_id', $dtr_id)
        ->where('time_in', 'like', "%".Carbon::now()->format('Y-m-d')."%")
        ->get();
        
        return $tl; 
    }

    public static function timeIn($dtr_id)
    {
        $timelogProcess = new Timelog();
        $timein = Carbon::now()->setTimezone(self::timezone($dtr_id));
        $logintime = Carbon::parse(Auth::user()->logintime)->shiftTimezone(self::timezone($dtr_id));
        if ($timein > $logintime) {
            $total_mins_tardy = Carbon::parse($timein)->diffInMinutes($logintime);
        } else {
            $total_mins_tardy = 0;
        }

        $timelogProcess->user_id = $dtr_id;
        $timelogProcess->time_in = $timein;
        $timelogProcess->total_mins_tardy = $total_mins_tardy;

        $timelogProcess->save();

        return $timelogProcess;
    }

    public static function timeOut($timelog_id) 
    {
        $updateTimelog = Timelog::find($timelog_id);
        if($updateTimelog->time_in) {
            $updateTimelog->time_out = Carbon::now()->setTimezone(self::timezone($dtr_id))->format('Y-m-d H:i:s');
            $updateTimelog->update();
        }
        return $updateTimelog;
    }

    public static function startBreak($data)
    {
        $breakProcess = new TimelogBreak;
        $breakProcess->timelog_id = $data['timelog_id'];
        $breakProcess->break_start = $data['break_start'];
        $breakProcess->notes = $data['notes'];
        $breakProcess->save();

        return $breakProcess;
    }

    public static function endBreak($timelog_id, $break_end)
    {
        $breaksChecker = TimelogBreak::with(['timelog'])->where([
            ['timelog_id', $timelog_id],
        ])->whereDate('break_start', Carbon::now()->format('Y-m-d'))->orderBy('break_start', 'DESC')->first();

        $totalPerBreaks = $breaksChecker->break_start->diffInMinutes($break_end);

        if ($breaksChecker->break_end === null) {
            $breaksChecker->break_end = $break_end;
            $breaksChecker->total_breaks = $totalPerBreaks;
            $breaksChecker->timelog['total_mins_break'] = $breaksChecker->timelog['total_mins_break'] + $totalPerBreaks;
            $breaksChecker->update();
            $breaksChecker->timelog->update();
            
            return $breaksChecker;
        }
        
        return false;
    }


    // Timelog Statistics
    public static function statistics($dtr_id)
    {
        Carbon::setWeekStartsAt(Carbon::MONDAY);
        Carbon::setWeekEndsAt(Carbon::FRIDAY);
        $from_date = Carbon::now()->startOfMonth()->shiftTimezone(self::timezone($dtr_id));
        $to_date = Carbon::now()->setTimezone(self::timezone($dtr_id));
        $tl = [];
        $abs_ctr = 0;
        $timelogStats = Timelog::with('breaks')->where('user_id', $dtr_id)->whereBetween('time_in', [Carbon::now()->format('Y-m-01 00:00:00'), Carbon::now()->format('Y-m-t 23:59:59')])->orderBy('time_in', 'DESC')->get();
        $user_timelog_week = Timelog::where('user_id', $dtr_id)->whereBetween('time_in', [Carbon::now()->startOfWeek()->shiftTimezone(self::timezone($dtr_id)), Carbon::now()->endOfWeek()->shiftTimezone(self::timezone($dtr_id))])->get();
        $todayLogin = Timelog::where('user_id', $dtr_id)->orderBy('time_in', 'DESC')->first();
        $tardyCountWk = $tardyCountMos = $weeklyWork = 0;
        
        // Absent Checker
        while (strtotime($from_date) <= strtotime($to_date)) {
            $date = $from_date->format('Y-m-d');
            $day = $from_date->format('D');
            if (!in_array($day, ['Sat', 'Sun'])) {
                $isPresent = 0;
                foreach ($timelogStats as $key => $value) {
                    if ($date == Carbon::createFromFormat('Y-m-d H:i:s', $value->time_in)->shiftTimezone(self::timezone($dtr_id))->format('Y-m-d')) {
                        $isPresent = 1;
                    }
                }
                if ($isPresent != 1) {
                    $tl[$date] = 'ABSENT';
                    if ($tl[$date] == 'ABSENT') {
                        $abs_ctr = $abs_ctr + 1;
                    }
                }
            }
            $from_date = $from_date->addDay();
        }
        $today_work_hours = null;
        // Total Weekly Work Hour
        foreach ($user_timelog_week as $weekly_total) {
            if($weekly_total->time_in->shiftTimezone(self::timezone($dtr_id))->format('Y-m-d') == $to_date->shiftTimezone(self::timezone($dtr_id))->format('Y-m-d')) {
                if ($weekly_total->time_out) {
                    $to = $weekly_total->time_out->shiftTimezone(self::timezone($dtr_id));
                } else {
                    $to = $to_date->shiftTimezone(self::timezone($dtr_id));
                }
                $today_work_hours = $weekly_total->time_in->shiftTimezone(self::timezone($dtr_id))->diffInSeconds($to);
            }
            if ($weekly_total->time_out != NULL) {
                $total_diff_sec = $weekly_total->time_in->shiftTimezone(self::timezone($dtr_id))->diffInSeconds($weekly_total->time_out->shiftTimezone(self::timezone($dtr_id)));
            } else {
                $total_diff_sec = $weekly_total->time_in->shiftTimezone(self::timezone($dtr_id))->diffInSeconds($to_date->format('H:i:s'));
            }
            $weeklyWork = ($total_diff_sec + $weeklyWork);
        }

        
        // Total Tardy Hours
        foreach ($timelogStats as $month_total) {
            $tardyCountMos = $month_total->total_mins_tardy + $tardyCountMos;
        }
        
        $displayStats = [
            'weekly_work_hours' => self::time_from_seconds($weeklyWork),
            'lates' => $tardyCountMos,
            'absents' => $abs_ctr,
            'today_work_hours' => self::time_from_seconds($today_work_hours)
        ];

        return $displayStats;

    }


    // Individual User Summary Report
    public static function summary_list($from=null, $to=null, $dtr_id)
    {
        $findTimelog = Timelog::select(['id', 'user_id', 'time_in', 'time_out', 'total_mins_break', 'total_mins_tardy'])->where('user_id', $dtr_id);
        if ($from && $to) {
            $findTimelog = $findTimelog->whereBetween('time_in', [$from, Carbon::parse($to)->addDay(1)->format('Y-m-d')])->orderBy('time_in', 'DESC');
        } else {
             $findTimelog = $findTimelog->whereBetween('time_in', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->orderBy('time_in', 'DESC');
        }

        return $findTimelog;
    }

    // Admin Summary Report
    public static function admin_summary($dtr_id)
    {
        $userArr = [];
        $userArr = User::select(['id', 'login_name', 'firstname', 'lastname'])->get();
        \JavaScript::put([
            'users' => $userArr,
        ]);
        return view('reports.timelog-admin-summary-list');

    }

    public static function admin_summary_list($userid, $admin_from=null, $admin_to=null, $dtr_ids)
    {

            $reportTimelog = [];

            if ($userid != 'all') {
                $allUser = User::where('id', $userid)->get();
            } else {
                $allUser = User::whereIn('id', $dtr_ids)->get();
            }

            foreach ($allUser as $key_user => $value_user) {
                $userTimelog = Timelog::select(['id', 'user_id', 'time_in', 'time_out', 'total_mins_break', 'total_mins_tardy'])->where('user_id', $value_user->id)->orderBy('time_in', 'DESC');

                if ($admin_from && $admin_to) {
                    $userTimelog = $userTimelog->whereBetween('time_in', [$admin_from, Carbon::parse($admin_to)->addDay(1)->format('Y-m-d')])->get();
                } else {
                    $userTimelog = $userTimelog->whereBetween('time_in', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->get();
                }

                foreach ($userTimelog as $key_timelog => $value_timelog) {
                    $reportTimelog[] = [
                        'user_id' => $value_user->id,
                        'user_name' => $value_user->firstname . " " . $value_user->lastname,
                        'date' => Carbon::parse($value_timelog->time_in)->format('M d, Y'),
                        'day' => Carbon::parse($value_timelog->time_in)->format('D'),
                        'in' => Carbon::parse($value_timelog->time_in)->shiftTimezone(self::timezone($value_user->id))->format('H:i:s'),
                        'out' => $value_timelog->time_out ? Carbon::parse($value_timelog->time_out)->shiftTimezone(self::timezone($value_user->id))->format('H:i:s') : null,
                        'time_in' => $value_timelog->time_in->shiftTimezone(self::timezone($value_user->id)),
                        'time_out' => $value_timelog->time_out ? Carbon::parse($value_timelog->time_out)->shiftTimezone(self::timezone($value_user->id)) : null,
                        'hours_worked' => $value_timelog->time_out ? $value_timelog->time_in->shiftTimezone(self::timezone($value_user->id))->diffInSeconds($value_timelog->time_out->shiftTimezone(self::timezone($value_user->id))) : null,
                        'total_mins_break' => $value_timelog->total_mins_break,
                        'total_mins_tardy' => $value_timelog->total_mins_tardy,
                    ];
                }
            }
        return $reportTimelog;
    }

    public static function getUsers($dtr_ids)
    {
        $slct_user = User::where('active', 1)->whereIn('id', $dtr_ids)->orderBy('firstname', 'asc')->get();
        return $slct_user;
    }

}
