<?php

namespace App\Http\Controllers\Timelog;

use App\Http\Controllers\Controller;
use App\Models\DTR\Timelog;
use App\Models\DTR\TimelogBreak;
use App\Models\DTR\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DtrCustomTimelogController extends Controller
{
    public static function getCollection($dtr_ids, $request)
    {
        if ($request->sort) {
            $sortBy = $request->sort == 'date' ? 'time_in' : $request->sort;
            $dir = $request->direction;
        }else{
            $sortBy = 'time_in';
            $dir = 'desc';
        }

        if (isset($request->search)) {
            $search = $request->search;
            $timelog = Timelog::with('user','breaks')->orWhere('time_in', 'like', '%' . $search . '%')->orWhereHas('user', function ($q) use ($search) {
                $q->where('firstname', 'like', '%'.$search.'%'); // firstname
                $q->orWhere('lastname', 'like', '%'.$search.'%'); // lastname
            })->whereHas('user', function ($q) {
                $q->where('active', 1); // active users  only
            })->whereIn('user_id', $dtr_ids)->orderBy($sortBy, $dir)->paginate($request->limit);
        } else {
            $timelog = Timelog::with('user','breaks')->whereHas('user', function ($q) {
                $q->where('active', 1); // active users only
            })->whereIn('user_id', $dtr_ids)->orderBy($sortBy, $dir)->paginate($request->limit);
        }
        // $userTimelogs = Timelog::with('user','breaks')->whereIn('user_id', $dtr_id)->orderBy('id', 'desc')->get();
        return $timelog;
    }

    public static function storeTimelog($dtr_id, $requestData){
        $request = $requestData;
        $user = User::where('id', $dtr_id)->first();
        //for tardy checker
        $userLogintime = Carbon::parse($user->logintime)->format('H:i:s');
        $timein = Carbon::parse($request['time_in'])->format('H:i:s');
        //duplicate checker
        $timelog = Timelog::where('user_id', $user->id)->get();
        foreach ($timelog as $key => $t) {
            if(Carbon::parse($t->time_in)->format('Y-m-d') == Carbon::parse($request['time_in'])->format('Y-m-d')){
                return ['status' => 'duplicate'];
            }
        }
        try {
            // start db transaction
            DB::beginTransaction();

            $tl = new Timelog;
            $tl->user_id = $user->id;
            $tl->time_in = $request['time_in'];
            $tl->time_out = $request['time_out'];
            $tl->total_mins_tardy = $userLogintime > $timein ? 0 : Carbon::parse($user->logintime)->diffInMinutes($timein);
            $tl->save();
            
            DB::commit();

            return ['status' => 'success', 'data' => $tl];

        } catch (\Exception $e) {
            // rollback db transactions
            DB::rollBack();

            // return to previous page with errors
            return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
        }
    }

    public static function updateTimelog($dtr_id, $requestData){
        $request = $requestData;
        $user = User::where('id', $dtr_id)->first();
        //for tardy checker
        $userLogintime = Carbon::parse($user->logintime)->format('H:i:s');
        $timein = Carbon::parse($request['time_in'])->format('H:i:s');
         //duplicate checker
         $timelog = Timelog::where('user_id', $user->id)->get();
         foreach ($timelog as $key => $t) {
             if($t->id != $request->id){
                 if(Carbon::parse($t->time_in)->format('Y-m-d') == Carbon::parse($request['time_in'])->format('Y-m-d')){
                     return ['status' => 'duplicate'];
                 }
             }
         }

        try {
            // start db transaction
            DB::beginTransaction();

            $tl = Timelog::with('breaks')->where('id', $request->id)->first();
            $tl->user_id = $user->id;
            $tl->time_in = $request['time_in'];
            $tl->time_out = $request['time_out'];
            $tl->total_mins_tardy = $userLogintime > $timein ? 0 : Carbon::parse($user->logintime)->diffInMinutes($timein);
            $tl->update();

            DB::commit();

            return ['status' => 'success', 'data' => $tl];

        } catch (\Exception $e) {
            // rollback db transactions
            DB::rollBack();

            // return to previous page with errors
            return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
        }
    }

    public static function getUserTimelogDtr($user_id, $timelog_id){
        $data = Timelog::with('breaks')->where('id', $timelog_id)->first();
        return $data;
    }

    public function storeBreak($requestData){
        $request = $requestData;
        try {
            // start db transaction
            DB::beginTransaction();

            $break = new TimelogBreak;
            $break->timelog_id = $request->timelog_id;
            $break->break_start = $request['break_start'];
            $break->break_end = $request['break_end'];
            $break->total_breaks = Carbon::parse($request['break_start'])->diffInMinutes(Carbon::parse($request['break_end']));
            $break->notes = $request->notes;
            $break->save();

            $this->updateTotalBreaks($request->timelog_id);

            DB::commit();

            return ['status' => 'success', 'data' => $break];

        } catch (\Exception $e) {
            // rollback db transactions
            DB::rollBack();

            // return to previous page with errors
            return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
        }
    }

    public function updateBreak($requestData){
        $request = $requestData;
        try {
            // start db transaction
            DB::beginTransaction();

            $break = TimelogBreak::where('id', $request->id)->first();
            $break->timelog_id = $request->timelog_id;
            $break->break_start = $request['break_start'];
            $break->break_end = $request['break_end'];
            $break->total_breaks = Carbon::parse($request['break_start'])->diffInMinutes(Carbon::parse($request['break_end']));
            $break->notes = $request->notes;
            $break->save();

            $this->updateTotalBreaks($break->timelog_id);

            DB::commit();

            return ['status' => 'success', 'data' => $break];

        } catch (\Exception $e) {
            // rollback db transactions
            DB::rollBack();

            // return to previous page with errors
            return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
        }
    }

    public function updateTotalBreaks($id){
        if(isset($id) && $id !== null){
            $timelog = Timelog::with('breaks')->where('id', $id)->first();
            $totalBreaks = $timelog->breaks->sum('total_breaks');
            $timelog->total_mins_break = $totalBreaks;
            $timelog->update();

        }
    }

    public function destroyTimelogBreak($dtr_id, $id)
    {
        $tlb = TimelogBreak::find($id);
        if ($tlb != null) {
            $tlb->delete();
            $this->updateTotalBreaks($tlb->timelog_id);
            return ['status' => 'success', 'data' => $id];
        }
    }

    public static function destroyTimelog($dtr_id, $id)
    {
        $tl = Timelog::find($id);
        if ($tl != null) {
            $tl->delete();
            return ['status' => 'success', 'data' => $id];
        }
    }

    public static function getUsers($dtr_ids){
        $slct_user = User::where('active', 1)->whereIn('id', $dtr_ids)->orderBy('firstname', 'asc')->get();
        return $slct_user;
    }
}
