<?php

namespace App\Http\Controllers\Timelog;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;

use Auth;
use App\Models\User;
use App\Models\Timelog;
use App\Models\TimelogBreak;

use App\Http\Requests\StoreTimelogRequest;
use App\Http\Requests\StoreTimelogBreakRequest;

class CustomTimelogController extends Controller
{
    public static function apiRoutes()
    {
        Route::middleware('auth')->group(function () {
            // list
            Route::get('timelog/get', [CustomTimelogController::class, 'getTimelogs']);
            Route::get('timelog/list', [CustomTimelogController::class, 'index']);
            // create
            Route::get('timelog/create', [CustomTimelogController::class, 'create']);
            Route::post('timelog/save', [CustomTimelogController::class, 'store']);
            // details
            Route::get('timelog/details/{user_id}/{timelog_id}', [CustomTimelogController::class, 'show']);
            Route::post('timelog/update', [CustomTimelogController::class, 'update']);
            Route::get('timelog/get/{user_id}/{timelog_id}', [CustomTimelogController::class, 'getUserTimelog']);
            Route::delete('timelog/delete/{user_id}/{id}', [CustomTimelogController::class, 'destroy']);
            // breaks
            Route::post('timelog/break/store', [CustomTimelogController::class, 'storeUpdateBreak']);
            Route::delete('timelog/break/delete/{user_id}/{id}', [CustomTimelogController::class, 'destroyBreak']);
            
        });
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        if(Auth::user()->hasRole(['Admin','Super-Admin'])){
            return view('timelog.admin.index');
        }else{
            abort(403, 'Unauthorized action.');
        }
        
    }

    public function getTimelogs(Request $request){

        $timelog = [];
        $dtr_ids = [];

        if(Auth::user()->dtr_id) {
            $allUser = User::with('party')->get();
            foreach ($allUser as $k => $kuser) {
                if($kuser->dtr_id){
                    array_push($dtr_ids, $kuser->dtr_id);
                }
            }
            $timelog = DtrCustomTimelogController::getCollection($dtr_ids, $request);
        }else{
            if ($request->sort) {
                $sortBy = $request->sort == 'date' ? 'time_in' : $request->sort;
                $dir = $request->direction;
            }else{
                $sortBy = 'time_in';
                $dir = 'desc';
            }
    
           if (isset($request->search)) {
                $search = $request->search;
                $timelog = Timelog::with('user.party','breaks')->orWhere('time_in', 'like', '%' . $search . '%')->orWhereHas('user.party', function ($q) use ($search) {
                    $q->where('name', 'like', '%'.$search.'%'); // user name
                })->whereHas('user', function ($q) {
                    $q->where('is_active', 1); // active users  only
                })->orderBy($sortBy, $dir)->paginate($request->limit);
            } else {
                $timelog = Timelog::with('user.party','breaks')->whereHas('user', function ($q) {
                    $q->where('is_active', 1); // active users only
                })->orderBy($sortBy, $dir)->paginate($request->limit);
            }

        }

        return $timelog;
        

    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if(Auth::user()->dtr_id){
            $slct_user = $this->getDtrUsers();
        }else{
            $slct_user = User::with('party')->where('is_active', 1)->whereNull('dtr_id')->orderBy('username', 'asc')->get();
        }

        \JavaScript::put([
            'slct_user' => $slct_user,
        ]);
        return view('timelog.admin.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreTimelogRequest $request)
    {

        $validated = $request->validated();

        if(Auth::user()->dtr_id){
            $tl = DtrCustomTimelogController::storeTimelog($validated['user_id'], $request);
            if($tl['status'] == 'success'){
                return ['status' => 'success', 'data' => $tl['data']];
            }elseif($tl['status'] == 'duplicate'){
                return ['status' => 'errors', 'data' => $request, 'message' => 'Time-in already exist.'];   
            }
        }else{
            
            $user = User::where('id', $validated['user_id'])->first();
            //for tardy checker
            $userLogintime = Carbon::parse($user->logintime)->format('H:i:s');
            $timein = Carbon::parse($validated['time_in'])->format('H:i:s');
             //duplicate checker
            $timelog = Timelog::where('user_id', $user->id)->get();
            foreach ($timelog as $key => $t) {
                if(Carbon::parse($t->time_in)->format('Y-m-d') == Carbon::parse($validated['time_in'])->format('Y-m-d')){
                    return ['status' => 'errors', 'data' => $request, 'message' => 'Time-in already exist.'];
                }
            }

            try {
                // start db transaction
                DB::beginTransaction();

                $tl = new Timelog;
                $tl->user_id = $user->id;
                $tl->time_in = $validated['time_in'];
                $tl->time_out = $validated['time_out'];
                $tl->total_mins_tardy = $userLogintime > $timein ? 0 : Carbon::parse($user->logintime)->diffInMinutes($timein);
                $tl->save();
                
                DB::commit();
    
                return ['status' => 'success', 'data' => $tl];
    
            } catch (\Exception $e) {
                // rollback db transactions
                DB::rollBack();
    
                // return to previous page with errors
                return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
            }
        }

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(StoreTimelogRequest $request)
    {

        $validated = $request->validated();

        if(Auth::user()->dtr_id) {
            $tl = DtrCustomTimelogController::updateTimelog($validated['user_id'], $request);
            if($tl['status'] == 'success'){
                return ['status' => 'success', 'data' => $tl['data']];
            }elseif($tl['status'] == 'duplicate'){
                return ['status' => 'errors', 'data' => $request, 'message' => 'Time-in already exist.'];   
            }
        } else { 
            try {
                // start db transaction
                DB::beginTransaction();
                
                $user = User::where('id', $validated['user_id'])->first();
                $userLogintime = Carbon::parse($user->logintime);
                $timein = Carbon::parse($validated['time_in'])->format('H:i:s');
                 //duplicate checker
                $timelog = Timelog::where('user_id', $user->id)->get();
                foreach ($timelog as $key => $t) {
                    if($t->id != $request->id){
                        if(Carbon::parse($t->time_in)->format('Y-m-d') == Carbon::parse($validated['time_in'])->format('Y-m-d')){
                            return ['status' => 'errors', 'data' => $request, 'message' => 'Time-in already exist.'];
                        }
                    }
                }

                $tl = Timelog::with('breaks')->where('id', $request->id)->first();
                $tl->user_id = $user->id;
                $tl->time_in = $validated['time_in'];
                $tl->time_out = $validated['time_out'];
                $tl->total_mins_tardy = $userLogintime > $timein ? $userLogintime->diffInMinutes($timein) : 0;
                $tl->update();
    
                DB::commit();
    
                return ['status' => 'success', 'data' => $tl];
    
            } catch (\Exception $e) {
                // rollback db transactions
                DB::rollBack();
    
                // return to previous page with errors
                return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
            }
        }


    }
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($user_id, $timelog_id)
    {

        if(Auth::user()->dtr_id){
            $slct_user = $this->getDtrUsers();
            $userTimelog = DtrCustomTimelogController::getUserTimelogDtr($user_id, $timelog_id);
        }else{
            $slct_user = User::with('party')->where('is_active', 1)->whereNull('dtr_id')->orderBy('username', 'asc')->get();
            $userTimelog = Timelog::with('breaks')->where('id', $timelog_id)->first();
        }

        \JavaScript::put([
            'slct_user'     => $slct_user,
            'timelog_id'    => $timelog_id,
            'user_id'       => $user_id,
            'userTimelog'   => $userTimelog,
        ]);

        return view('timelog.admin.show');
    }

    public function getUserTimelog($user_id, $timelog_id)
    {

        if(Auth::user()->dtr_id){
            $data = DtrCustomTimelogController::getUserTimelogDtr($user_id, $timelog_id);
            return $data;
        }else{
            $data = Timelog::with('breaks')->where('id', $timelog_id)->first();
            return $data;
        }
        
    }

    public function storeUpdateBreak(StoreTimelogBreakRequest $request) {

        $validated = $request->validated();

        if($request->edit == false){
            try {
                // start db transaction
                DB::beginTransaction();

                if(Auth::user()->dtr_id){
                    $tl = new DtrCustomTimelogController;
                    $tl = $tl->storeBreak($request);
                    if($tl['status'] == 'success'){
                        return ['status' => 'success', 'data' => $tl['data']];
                    }
                }

                $break = new TimelogBreak;
                $break->timelog_id = $request->timelog_id;
                $break->break_start = $validated['break_start'];
                $break->break_end = $validated['break_end'];
                $break->total_breaks = Carbon::parse($validated['break_start'])->diffInMinutes(Carbon::parse($validated['break_end']));
                $break->notes = $request->notes;
                $break->save();

                $this->updateTotalBreaks($request->timelog_id);

                DB::commit();
    
                return ['status' => 'success', 'data' => $break];
    
            } catch (\Exception $e) {
                // rollback db transactions
                DB::rollBack();
    
                // return to previous page with errors
                return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
            }
        }else{
            try {
                // start db transaction
                DB::beginTransaction();
                
                if(Auth::user()->dtr_id){
                    $tl = new DtrCustomTimelogController;
                    $tl = $tl->updateBreak($request);
                    if($tl['status'] == 'success'){
                        return ['status' => 'success', 'data' => $tl['data']];
                    }
                }

                $break = TimelogBreak::where('id', $request->id)->first();
                $break->break_start = $validated['break_start'];
                $break->break_end = $validated['break_end'];
                $break->total_breaks =  Carbon::parse($validated['break_start'])->diffInMinutes(Carbon::parse($validated['break_end']));
                $break->notes = $request->notes;
                $break->update();

                $this->updateTotalBreaks($break->timelog_id);

                DB::commit();
    
                return ['status' => 'success'];
    
            } catch (\Exception $e) {
                // rollback db transactions
                DB::rollBack();
    
                // return to previous page with errors
                return json_encode(['data' => $request, 'message' => $e->getMessage(), 'status' => 'errors']);
            }
        }
    }

    public function destroyBreak($user_id, $id)
    {
        if(Auth::user()->dtr_id){
            $tlb = new DtrCustomTimelogController;
            $tlb = $tlb->destroyTimelogBreak($user_id, $id);
            if($tlb['status'] == 'success'){
                return ['status' => 'success', 'id' => $id];
            }
        }else{
            $tlb = TimelogBreak::find($id);
            if ($tlb != null) {
                $tlb->delete();
                $this->updateTotalBreaks($tlb->timelog_id);
                return json_encode(['id' => $id, 'status' => 'success']);
            }
        }
        
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($user_id, $id)
    {
        if(Auth::user()->dtr_id){
            $tl = DtrCustomTimelogController::destroyTimelog($user_id, $id);
            if($tl['status'] == 'success'){
                return ['status' => 'success', 'id' => $id];
            }
        }else{
            $tl = Timelog::find($id);
            if ($tl != null) {
                $tl->delete();
                return json_encode(['id' => $id, 'status' => 'success']);
            }
        }
        
    }

    public function updateTotalBreaks($id){
        if(isset($id) && $id !== null){
            $timelog = Timelog::with('breaks')->where('id', $id)->first();
            $totalBreaks = $timelog->breaks->sum('total_breaks');
            $timelog->total_mins_break = $totalBreaks;
            $timelog->update();

        }
    }

    public function getDtrUsers(){
        $dtr_ids = [];
        $allUser = User::with('party')->get();
        foreach ($allUser as $k => $kuser) {
            if($kuser->dtr_id){
                array_push($dtr_ids, $kuser->dtr_id);
            }
        }
        $slct_user = DtrCustomTimelogController::getUsers($dtr_ids);
        return $slct_user;
    }

}
