<?php

namespace App\Http\Controllers\Online\ClientInquiry;

use App\Http\Controllers\Deals\ClientController;
use App\Http\Controllers\Controller;
use App\Models\ClientDetail;
use App\Models\ContactDetail;
use App\Models\Deal;
use App\Models\Party;
use App\Models\Person;
use App\Models\DealStatus;
use App\Models\VisaStatus;
use App\Models\Organisation;
use App\Models\DealNote;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ClientInquiryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if(config('app.name') != 'KAUR') {
            abort(404);
        }

        \JavaScript::put([
            'visas' => VisaStatus::all(),
        ]);

        return view('forms.client-form.index');
    }

        /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        // dd($request->all());

        $attributes = [
            'firstname' => 'First Name',
            'lastname' => 'Last Name',
            'date_of_birth' => 'Date of Birth',
            'phone_1' => 'Mobile Number',
            'email' => 'Email Address',
            'gender' => 'Gender',
            'address' => 'Address',
            'state' => 'State',
            'postcode' => 'Postcode',
            'home_country_res_addr' => 'Home Country Address',
            'visa_status' => 'Current Visa Status',
            // 'college_currently_enroled' => 'College Enrolled Already',
        ];

        $validate = Validator::make($request->all(), [
            'firstname' => 'required',
            'lastname' => 'required',
            'date_of_birth' => 'required',
            'phone_1' => 'required',
            'email' => 'required|email:rfc,dns',
            'gender' => 'required',
            'address' => 'required',
            'state' => 'required',
            'postcode' => 'required',
            'home_country_res_addr' => 'required',
            'visa_status' => 'required',
            // 'college_currently_enroled' => 'required',
        ], [], $attributes);
        
        if ($validate->fails()) {
            return response()->json(['status' => 'errors', 'errors' => $validate->messages()]);
        }
        
        try {

            $party = new Party;
            $person = new Person;
            $contact_detail = new ContactDetail;
            $deal = new Deal;
            $client_detail = new ClientDetail;

            DB::beginTransaction();
            $party->fill([
                'party_type_id'    => 1,
                'name'          => preg_replace('/\s+/', ' ', $request->firstname . ' ' . $request->middlename . ' ' . $request->lastname),
                // 'active'        => 1
            ]);

            $party->save();

            $person->fill([
                'firstname'     => $request->firstname,
                'middlename'    => isset($request->middlename) ? $request->middlename : null,
                'lastname'      => $request->lastname,
                'date_of_birth' => isset($request->date_of_birth) ? Carbon::parse($request->date_of_birth)->setTimezone('Australia/Melbourne')->format('Y-m-d') : null,
                'gender'        => isset($request->gender) ? $request->gender : null,
                'prefix'        => isset($request->prefix) ? $request->prefix : null,
                'person_type_id'   => 5
            ]);
            $person->party()->associate($party);
            $person->save();


            $phones = [];
            // phone 1
            if(isset($request->phone_1)){
                $phones[] = $request->phone_1;
            }
            // phone 2
            if(isset($request->phone_2)){
                $phones[] = $request->phone_2;
            }
            // phone 3
            if(isset($request->phone_3)){
                $phones[] = $request->phone_3;
            }

            $contact_detail->fill([
                'address_1' => isset($request->address) ? $request->address : null,
                'country_id' => isset($request->country['id']) ? $request->country['id'] : null,
                'state' => isset($request->state) ? $request->state : null,
                'postcode' => isset($request->postcode) ? $request->postcode : null,
                'phones' => count($phones) > 0 ? $phones : null,
                // 'phone_1' => isset($request->phone_1) ? $request->phone_1 : null,
                // 'phone_2' => isset($request->phone_2) ? $request->phone_2 : null,
                // 'phone_3' => isset($request->phone_3) ? $request->phone_3 : null,
                'email' => isset($request->email) ? $request->email : null,
                'email_alt' => isset($request->email_alt) ? $request->email_alt : null,
                'home_country_res_addr' => isset($request->home_country_res_addr) ? $request->home_country_res_addr : null,
            ]);
            $contact_detail->person()->associate($person);
            $contact_detail->save();
            
            $deal->fill([
                'deal_type_id' => isset($request->deal_type_id) ? $request->deal_type_id : 1,
                'deal_source_id' => isset($request->source['id']) ? $request->source['id'] : null,
                'is_test' => isset($request->is_test) ? $request->is_test : 0,
                'is_online_inquiry' => isset($request->is_online_inquiry) ? $request->is_online_inquiry : 0,
                'comments' => isset($request->comments) ? $request->comments : null
            ]);
            $deal->party()->associate($party);
            $deal->user()->associate(Auth::user());
            $deal->deal_status()->associate(DealStatus::find(1));
            $deal->save();

            $deal->active_date = Carbon::parse($deal->created_at)->format('Y-m-d');
            $deal->update();


            $c_ID = new ClientController;
            $client_detail->fill([
                'customer_id' => $c_ID->generate_customer_id(),
                'current_visa_status'       => isset($request->visa_status['id']) ? $request->visa_status['id'] : null,
                'college_currently_enroled' => isset($request->college_currently_enroled) ? $request->college_currently_enroled : null
            ]);
            $client_detail->deal()->associate($deal);
            $client_detail->save();

            // update client_detail_id in contact_details table
            $contact_detail->client_detail()->associate($client_detail);
            $contact_detail->update();

            $note = new DealNote;
            $note->fill([
                'remarks' => isset($request->comments) ? $request->comments : null,
                'note_type' => 'convo',
                'note_date' => Carbon::now()->setTimezone('Australia/Melbourne')->format('Y-m-d H:i:s')
            ]);
            $note->deal()->associate($deal);
            $note->save();


            DB::commit();


            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();
            throw $th;
        }

    }
}