<?php

namespace App\Http\Controllers\Deals;

use App\Http\Controllers\Controller;
use App\Http\Resources\ClientResource;
use App\Models\Agent;
use App\Models\AvtCountryIdentifier;
use App\Models\ClientDetail;
use App\Models\ClientReminder;
use App\Models\ContactDetail;
use App\Models\Country;
use App\Models\Deal;
use App\Models\DealAttachment;
use App\Models\DealAttachmentFolder;
use App\Models\DealNote;
use App\Models\DealSource;
use App\Models\DealStatus;
use App\Models\DealType;
use App\Models\Organisation;
use App\Models\Party;
use App\Models\PaymentMethod;
use App\Models\Person;
use App\Models\ServiceType;
use App\Models\User;
use App\Models\VisaStatus;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ClientController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        \JavaScript::put([
            'countries' => AvtCountryIdentifier::all(),
            'sources' => DealSource::all(),
            'userRole' => Auth::user()->roles[0]->name, 
        ]);

        return view('clients.index');
    }

    public function client_query()
    {
        //
        \JavaScript::put([
            'countries' => AvtCountryIdentifier::all(),
            'sources' => DealSource::all(),
            'userRole' => Auth::user()->roles[0]->name, 
        ]);

        return view('clients.query.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        // dd($request->all());

        $attributes = [
            'firstname' => 'First Name',
            'lastname' => 'Last Name',
            'date_of_birth' => 'Date of Birth',
            'phone_1' => 'Phone # 1',
            'email' => 'Email Address',
        ];

        if (!isset($request->deal_type_id) && $request->deal_type_id != 1) {
            $validate = Validator::make($request->all(), [
                'firstname' => 'required',
                'lastname' => 'required',
                'date_of_birth' => 'required',
                'phone_1' => 'required',
                'email' => 'required|email:rfc,dns',
            ], [], $attributes);
    
        }else {
            $validate = Validator::make($request->all(), [
                'firstname' => 'required',
                'lastname' => 'required',
            ], [], $attributes);
        }
        
        if ($validate->fails()) {
            return response()->json(['status' => 'errors', 'errors' => $validate->messages()]);
        }
        
        try {

            $party = new Party;
            $person = new Person;
            $contact_detail = new ContactDetail;
            $deal = new Deal;
            $client_detail = new ClientDetail;

            DB::beginTransaction();
            $party->fill([
                'party_type_id'    => 1,
                'name'          => preg_replace('/\s+/', ' ', $request->firstname . ' ' . $request->middlename . ' ' . $request->lastname),
                // 'active'        => 1
            ]);

            $party->save();

            $person->fill([
                'firstname'     => $request->firstname,
                'middlename'    => isset($request->middlename) ? $request->middlename : null,
                'lastname'      => $request->lastname,
                'date_of_birth' => isset($request->date_of_birth) ? Carbon::parse($request->date_of_birth)->setTimezone('Australia/Melbourne')->format('Y-m-d') : null,
                'gender'        => isset($request->gender) ? $request->gender : null,
                'prefix'        => isset($request->prefix) ? $request->prefix : null,
                'person_type_id'   => 5
            ]);
            $person->party()->associate($party);
            $person->save();


            $phones = [];
            // phone 1
            if(isset($request->phone_1)){
                $phones[] = $request->phone_1;
            }
            // phone 2
            if(isset($request->phone_2)){
                $phones[] = $request->phone_2;
            }
            // phone 3
            if(isset($request->phone_3)){
                $phones[] = $request->phone_3;
            }

            $contact_detail->fill([
                'address_1' => isset($request->address) ? $request->address : null,
                'country_id' => isset($request->country['id']) ? $request->country['id'] : null,
                'state' => isset($request->state) ? $request->state : null,
                'postcode' => isset($request->postcode) ? $request->postcode : null,
                'phones' => count($phones) > 0 ? $phones : null,
                // 'phone_1' => isset($request->phone_1) ? $request->phone_1 : null,
                // 'phone_2' => isset($request->phone_2) ? $request->phone_2 : null,
                // 'phone_3' => isset($request->phone_3) ? $request->phone_3 : null,
                'email' => isset($request->email) ? $request->email : null,
                'email_alt' => isset($request->email_alt) ? $request->email_alt : null,
                'home_country_res_addr' => isset($request->contact->home_country_res_addr) ? $request->contact->home_country_res_addr : null,
            ]);
            $contact_detail->person()->associate($person);
            $contact_detail->save();
            
            $deal->fill([
                'deal_type_id' => isset($request->deal_type_id) ? $request->deal_type_id : 2,
                'deal_source_id' => isset($request->source['id']) ? $request->source['id'] : null,
                'is_test' => isset($request->is_test) ? $request->is_test : 0,
                'is_online_inquiry' => isset($request->is_online_inquiry) ? $request->is_online_inquiry : 0
            ]);
            $deal->party()->associate($party);
            $deal->user()->associate(Auth::user());
            $deal->deal_status()->associate(DealStatus::find(1));
            $deal->save();

            $deal->active_date = Carbon::parse($deal->created_at)->format('Y-m-d');
            $deal->update();

            $client_detail->fill([
                'customer_id' => $this->generate_customer_id(),
                'nationality_id' => isset($request->country['id']) ? $request->country['id'] : null,
            ]);
            $client_detail->deal()->associate($deal);
            $client_detail->save();

            // update client_detail_id in contact_details table
            $contact_detail->client_detail()->associate($client_detail);
            $contact_detail->update();

            DB::commit();


            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();
            throw $th;
        }

    }

    // Update Deals
    public function update_deals(Request $request)
    {
        try {
            DB::beginTransaction();

            $deal = Deal::where('id', $request->deal['id'])->first();
            if(isset($request->deal['deal_source']['id'])) {
                $source = DealSource::where('id', $request->deal['deal_source']['id'])->first();
                $deal->deal_source()->associate($source);
            }
            if(isset($request->deal['deal_type']['id'])) {
                $type = DealType::where('id', $request->deal['deal_type']['id'])->first();
                $deal->deal_type()->associate($type);
            }
            if(isset($request->deal['deal_status']['id'])) {
                $status = DealStatus::where('id', $request->deal['deal_status']['id'])->first();
                $deal->deal_status()->associate($status);
            }

            // update is_test 
            $deal->is_test = isset($request->deal['is_test']) && !in_array($request->deal['is_test'], ['', null, false]) ? 1 : 0;

            // update active date
            $deal->active_date = isset($request->deal['active_date']) ? Carbon::parse($request->deal['active_date'])->setTimezone('Australia/Melbourne')->format('Y-m-d') : null;
            
            // update is_online_inquiry 
            $deal->is_online_inquiry = isset($request->deal['is_online_inquiry']) && !in_array($request->deal['is_online_inquiry'], ['', null, false]) ? 1 : 0;

            $deal->update();
            DB::commit();

            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();
            throw $th;
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
        // dd($id);

        $deal = Deal::with(['party.person.contact_detail.country', 'client_detail.nationality'])->where('id', $id)->first();
        $user_deal = Auth::user()->id == $deal->user_id ? true : false;
        // dd($user_deal);

        \JavaScript::put([

            'countries' => AvtCountryIdentifier::all(),
            'nationalities' => AvtCountryIdentifier::where('nationality', '!=', null)->get(),
            'sources' => DealSource::all(),
            'statuses' => DealStatus::all(),
            'paymentMethods' => PaymentMethod::all(),
            'types' => DealType::all(),
            'visas' => VisaStatus::all(),
            'agents' => Agent::with('party')->get(),
            'users' => User::with('party')->where('id', '!=', 1)->get(),
            'userRole' => Auth::user()->roles[0]->name,
            'currentUser' => Auth::user(),
            'deal_id' => $id,
            // 'party' => $client->party,
            // 'person' => $client->party->person,
            'service_types' => ServiceType::all(),
            'user_deal' => $user_deal,

        ]);

        return view('clients.show');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        // dd($id);
        //
        $client = Deal::with(['deal_type','deal_source','deal_status','party.person.contact_detail.country', 'client_detail.nationality', 'client_detail.visa_status', 'client_detail.referrer.deal.party.person.contact_detail', 'invoices.payments.user.party', 'invoices.payments.payment_method', 'invoices.invoice_details','services.deal', 'services.assigned_to'])->where('id', $id)->first();

        $data = [
            'client' => $client,
        ];

        return $data;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
        // dd($request->all());

        $attributes = [
            'person.firstname' => 'First Name',
            'person.lastname' => 'Last Name',
            'person.date_of_birth' => 'Date of Birth',
            'contact.phone_1' => 'Phone # 1',
            'contact.email' => 'Email Address',
        ];

        $validate = Validator::make($request->all(), [
            'person.firstname' => 'required',
            'person.lastname' => 'required',
            'person.date_of_birth' => 'required',
            'contact.phone_1' => 'required',
            'contact.email' => 'required|email:rfc,dns',
        ], [], $attributes);

        if ($validate->fails()) {
            return response()->json(['status' => 'errors', 'errors' => $validate->messages()]);
        }
        
        try {
            
            $client = Deal::with(['party.person.contact_detail.country', 'client_detail.nationality'])->where('id', $id)->first();
            

            DB::beginTransaction();
            $client->party->fill([
                'party_type_id'    => 1,
                'name'          => preg_replace('/\s+/', ' ', $request->person['firstname'] . ' ' . $request->person['middlename'] . ' ' . $request->person['lastname']),
                // 'active'        => 1
            ]);
            $client->party->update();

            $client->party->person->fill([
                'firstname'     => $request->person['firstname'],
                'middlename'    => isset($request->person['middlename']) ? $request->person['middlename'] : null,
                'lastname'      => $request->person['lastname'],
                'date_of_birth' => isset($request->person['date_of_birth']) ? Carbon::parse($request->person['date_of_birth'])->setTimezone('Australia/Melbourne')->format('Y-m-d') : null,
                'gender'        => isset($request->person['gender']) ? $request->person['gender'] : null,
                'prefix'        => isset($request->person['prefix']) ? $request->person['prefix'] : null,
            ]);
            $client->party->person->update();
            
            $phones = [];
            // phone 1
            if(isset($request->contact['phone_1'])){
                $phones[] = $request->contact['phone_1'];
            }
            // phone 2
            if(isset($request->contact['phone_2'])){
                $phones[] = $request->contact['phone_2'];
            }
            // phone 3
            if(isset($request->contact['phone_3'])){
                $phones[] = $request->contact['phone_3'];
            }

            $client->party->person->contact_detail->fill([
                'address_1' => isset($request->contact['address_1']) ? $request->contact['address_1'] : null,
                'country_id' => isset($request->contact['country']['id']) ? $request->contact['country']['id'] : null,
                'state' => isset($request->contact['state']) ? $request->contact['state'] : null,
                'postcode' => isset($request->contact['postcode']) ? $request->contact['postcode'] : null,
                'phones' => count($phones) > 0 ? $phones : null,
                // 'phone_1' => isset($request->contact['phone_1']) ? $request->contact['phone_1'] : null,
                // 'phone_2' => isset($request->contact['phone_2']) ? $request->contact['phone_2'] : null,
                // 'phone_3' => isset($request->contact['phone_3']) ? $request->contact['phone_3'] : null,
                'email' => isset($request->contact['email']) ? $request->contact['email'] : null,
                'email_alt' => isset($request->contact['email_alt']) ? $request->contact['email_alt'] : null,
                'emer_relationship' => isset($request->contact['emer_relationship']) ? $request->contact['emer_relationship'] : null,
                'emer_name' => isset($request->contact['emer_name']) ? $request->contact['emer_name'] : null,
                'emer_email' => isset($request->contact['emer_email']) ? $request->contact['emer_email'] : null,
                'emer_address' => isset($request->contact['emer_address']) ? $request->contact['emer_address'] : null,
                'emer_phone' => isset($request->contact['emer_phone']) ? $request->contact['emer_phone'] : null,
                'home_country_res_addr' => isset($request->contact['home_country_res_addr']) ? $request->contact['home_country_res_addr'] : null,
            ]);
            $client->party->person->contact_detail->update();
            // $contact_detail->person()->associate($person);
            // $contact_detail->save();
            
            // $deal->fill([
            //     'deal_type_id' => 1,
            //     'deal_source_id' => isset($request->source['id']) ? $request->source['id'] : null
            // ]);
            // $deal->party()->associate($party);
            // $deal->user()->associate(Auth::user());
            // $deal->save();

            // $client_detail->fill([
            //     'customer_id' => $this->generate_customer_id(),
            //     'nationality_id' => isset($request->country['id']) ? $request->country['id'] : null,
            // ]);
            // $client_detail->deal()->associate($deal);
            // $client_detail->save();

            DB::commit();


            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();
            throw $th;
        }

    }

    public function lead_to_prospect($id)
    {

        $deal = Deal::with(['party.person.contact_detail'])->where('id',$id)->first();
       
        if($deal){

            $check = [
                'Firstname' => $deal->party->person->firstname,
                'Lastname' => $deal->party->person->lastname,
                'Date of Birth' => $deal->party->person->date_of_birth,
                'Phone' => $deal->party->person->contact_detail->phones,
                'Email' => $deal->party->person->contact_detail->email,
            ];

            $errors = [];

            foreach($check as $k => $v) {
                if(in_array($v, ['', null])) {
                    $errors[] = $k;
                }
            }

            if(count($errors) > 0) {
                return ['status' => 'error', 'msg' => 'please update required field(s):<br><b><span style="color:red">'.implode(', ', $errors).'</span></b>'];
            }else{
                $deal->deal_type_id = 2;
                $deal->update();
            }


            return ['status' => 'success'];
        }else{
            return ['status' => 'error', 'msg' => 'no such client available'];
        }

    }

    public function update_client_detail(Request $request)
    {
        $detail = $request->detail;
        try {
            
            $client = Deal::with(['client_detail.nationality'])->where('id', $detail['deal_id'])->first();
            

            DB::beginTransaction();

            $client->client_detail->fill([
                'nationality_id'            => isset($detail['nationality']['id']) ? $detail['nationality']['id'] : null,
                'current_visa_status'       => isset($detail['visa_status']['id']) ? $detail['visa_status']['id'] : null,
                'expiration_date'           => isset($detail['expiration_date']) ? Carbon::parse($detail['expiration_date'])->setTimezone('Australia/Melbourne')->format('Y-m-d') : null,
                'passport_no'               => $detail['passport_no'],
                'fifth_call'                => $detail['fifth_call'] == 1 ? 1 : 0,
                'college_currently_enroled' => $detail['college_currently_enroled'],
                'referrer_customer_id'      => isset($detail['referrer']['customer_id']) ? $detail['referrer']['customer_id'] : null,
                'client_member_card_no'     => $detail['client_member_card_no'],
                'client_member'             => $detail['client_member'] == 1 ? 1 : 0,
                'agent_id'                  => isset($detail['agent']['id']) ? $detail['agent']['id'] : null,
                'assigned_to'               => isset($detail['assigned']['id']) ? $detail['assigned']['id'] : null,
            ]);
            $client->client_detail->update();

            DB::commit();


            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();
            throw $th;
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
        $client = Deal::with(['deal_type','deal_source','deal_status','party.person.contact_detail.country', 'client_detail.nationality', 'client_detail.visa_status', 'client_detail.referrer.deal.party.person.contact_detail', 'invoices.payments.user.party', 'invoices.invoice_details'])->where('id', $id)->first();

        if(!$client) {
            return ['status' => 'error', 'msg' => 'No such client existed.'];
        }

        try {
            DB::beginTransaction();

            // invoice & payments
            foreach ($client->invoices as $k => $v) {
                foreach ($v->payments as $kk => $vv) {
                    $vv->delete();
                }
                $v->delete();
            }

            // attachments & folders
            $att_folders = DealAttachmentFolder::where('deal_id', $id)->get();
            foreach ($att_folders as $k => $v) {
                $v->delete();
            }
            $att = DealAttachment::where('deal_id', $id)->get();
            foreach ($att as $k => $v) {
                $v->delete();
            }

            // notes
            $notes = DealNote::where('deal_id', $id)->get();
            foreach ($notes as $k => $v) {
                $v->delete();
            }
            
            // client reminders
            $reminders = ClientReminder::where('deal_id', $id)->get();
            foreach ($reminders as $k => $v) {
                $v->delete();
            }

            // client details
            if(isset($client->client_detail->id)) {
                $client->client_detail->delete();
            }
            

            // contact details
            if(isset($client->party->person->contact_detail->id)) {
                $client->party->person->contact_detail->delete();
            }

            // person
            if(isset($client->party->person->id)) {
                $client->party->person->delete();
            }

            // party
            if(isset($client->party->id)) {
                $client->party->delete();
            }

            // deal
            $client->delete();

            DB::commit();

            return ['status' => 'success'];

        } catch (\Throwable $th) {
            DB::rollback();

            return ['status' => 'error', 'msg' => $th];

            throw $th;
        }
        
        

    }

    public function list(Request $request)
    {
        // dd($request->all());
        $client = collect();

        if ($request->sort) {
            $sort = $request->sort;
            $direction = $request->direction;
            
            if (isset($request->search) && $request->search != null) {
                
                $search = $request->search;

                if (\Auth::user()->hasRole('Demo')) {
                    
                } else {
                    $client = Deal::with(['deal_type','party.person', 'client_detail.contact_detail']);
                    
                        if(isset($request->type)) {
                            $client->where('deal_type_id', $request->type);
                        }else {
                            $client->where('deal_type_id', '!=', 1);
                        }

                        if(isset($request->withTest) && $request->withTest == 'false') {
                            $client->where('is_test', 0);
                        }

                        $client->whereHas('party', function($q) use($search) {
                            $q->where('name', 'like', '%'.$search.'%');
                        });

                        if(isset($request->type)) {
                            $client->where('deal_type_id', $request->type);
                        }else {
                            $client->where('deal_type_id', '!=', 1);
                        }

                        $client->orWhereHas('client_detail', function($q) use($search) {
                            $q->where('customer_id', 'like', '%'.$search.'%');
                        });
                        
                        if(isset($request->type)) {
                            $client->where('deal_type_id', $request->type);
                        }else {
                            $client->where('deal_type_id', '!=', 1);
                        }

                        $client->orWhereHas('client_detail.contact_detail', function($q) use($search) {
                            $q->where('phones', 'like', '%'.$search.'%');
                            $q->orWhere('email', 'like', '%'.$search.'%');
                            // $q->orWhere('email_alt', 'like', '%'.$search.'%');
                        });
                        
                        if(isset($request->type)) {
                            $client->where('deal_type_id', $request->type);
                        }else {
                            $client->where('deal_type_id', '!=', 1);
                        }
                        // ->orWhereHas('deal_type', function($q) use($search) {
                        //     $q->where('type', 'like', '%'.$search.'%');
                        // });
                        // ->orderBy($request->sort, $request->direction);
                    
                    // dd($client->get());
                }
                // ->where('student_id', 'like', '%'.$request->search.'%')->where('name', 'like', '%'.$request->search.'%')->where('type', 'like', '%'.$request->search.'%');

                // $student = Student::where('student_id','like', '%' . $request->search . '%');
            } else {
                // dd($request->sort);
                if (\Auth::user()->hasRole('Demo')) {
                    
                } else {
                    $client = Deal::with(['deal_type','party.person', 'client_detail.contact_detail'])->select('deals.id as id', 'deals.party_id', 'deals.deal_type_id');

                    if(isset($request->type)) {
                        $client->where('deal_type_id', $request->type);
                    }else {
                        $client->where('deal_type_id', '!=', 1);
                    }

                    if(isset($request->withTest) && $request->withTest == 'false') {
                        $client->where('is_test', 0);
                    }

                    if($sort == 'deal_type'){
                        $client->join('deal_types', 'deal_types.id', '=', 'deals.deal_type_id')
                        ->orderBy('deal_types.type', $direction);
                    }
                    if($sort == 'name') {
                        $client->join('party', 'party.id', '=', 'deals.party_id')
                        ->orderBy('party.name', $direction);
                    }
                    if($sort == 'email') {
                        $client->join('client_details', 'client_details.deal_id', '=', 'deals.id')
                        ->join('contact_details', 'contact_details.client_detail_id', '=', 'client_details.id')
                        ->orderBy('contact_details.email', $direction);
                    }
                    if($sort == 'phone') {
                        $client->join('client_details', 'client_details.deal_id', '=', 'deals.id')
                        ->join('contact_details', 'contact_details.client_detail_id', '=', 'client_details.id')
                        ->orderBy('contact_details.phones', $direction);
                    }

                    if($sort == 'deals.created_at') {
                        $client->orderBy($sort, $direction);
                    }

                    if($sort == 'customer_id') {
                        $client->join('client_details', 'client_details.deal_id', '=', 'deals.id')
                        ->orderBy('client_details.customer_id', $direction);
                    }
                }
            }

        }
        
        
        // $cs = $client->paginate($request->limit);
        // foreach($cs as $k => $v){
        //     dump($v);
        // }
        // dd('end');
        
        return ClientResource::collection($client->paginate($request->limit));
    }

    public function generate_customer_id($customer_id = null)
    {
        $to = Organisation::first();
        // $client = null;
        // if($customer_id == null){
        //     $client = ClientDetail::where('customer_id', '!=', null)->latest()->first();
        //     $customer_id = $client ? $client->customer_id : null;
        // }
        $client = ClientDetail::where('customer_id', '!=', null)->latest()->first();
        $customer_id = $client ? $client->customer_id : null;
        $next_no = $customer_id != null ? abs(str_replace($to->client_id_prefix,"",$customer_id)) : 0;
        $next = 0;
        $next_id = '';
        while ($next != 1) {
            
            $prefix = in_array($to->client_id_prefix, [null, '']) ? 'VRX' : $to->client_id_prefix;
            $next_id = $prefix . str_pad($next_no , 5, 0, STR_PAD_LEFT);
            $check = ClientDetail::where('customer_id', $next_id)->first();
            
            if($check){
                $next_no++;
            }else {
                $next = 1;
            }
        }
        // dump('-> '.$next_id);
        return $next_id;
        // dump($next_id);
        
        // dd($check);
        // if(isset($check->id)) {
            // dump('check -> '. $check->customer_id);
        // } else{ 
            // dump('check -> none -> ' . $next_id);
        // }
        // dump('****************************************************');
        // if(isset($check->id)){
        //     $this->generate_customer_id($next_id);
        // }else {
        //     dump($next_id . '<-');
        //     return $next_id;
        // }
    }

    public function search_referrer($search)
    {
        $client = ClientDetail::with(['deal.party', 'contact_detail'])->whereHas('deal.party', function($q) use($search) {
            $q->where('name', 'like', '%'.$search.'%');
        })->orWhereHas('contact_detail', function($q) use($search) {
            $q->where('phones', 'like', '%'.$search.'%');
        })->orWhere('customer_id', 'like', '%'.$search.'%')->get();

        return $client;
    }

    public function top_search($search)
    {
        // dd(is_numeric($search));
        // dd($search);
        $client = ClientDetail::with(['deal.party' ,'contact_detail'])->whereHas('deal.party', function($q) use($search) {
            $q->where('name', 'like', '%'.$search.'%');
        })->orWhereHas('contact_detail', function($q) use($search) {
            $q->where('phones', 'like', '%'.$search.'%');
        })->orWhere('customer_id', 'like', '%'.$search.'%')->limit(20)->get();

        $clnt = [];
        foreach ($client as $val) {
            $phone = !in_array($val->contact_detail->phones, ['', null]) ? implode(', ', $val->contact_detail->phones) : '';
            $clnt[] = [
                'client_name' => $val->customer_id . ' - ' . $val->deal->party->name .' (' . $phone .')',
                'name' => $val->deal->party->name,
                'deal_id' => $val->deal->id,
                'phones' => $phone,
                // 'client_name' => $val->customer_id . ' - ' . $val->name,
                // 'name' => $val->name,
                // 'deal_id' => $val->deal_id,
            ];
        }
    
        return $clnt;
    }
}
